<?php
/**
 * AIHelper批量操作页面
 */

use Typecho\Db;
use Widget\Options;

$db = Db::get();

// 处理AJAX生成文章请求
if (isset($_POST['ajax_action']) && $_POST['ajax_action'] == 'generate_article') {
    header('Content-Type: application/json');

    $title = isset($_POST['title']) ? trim($_POST['title']) : '';

    if (empty($title)) {
        echo json_encode(['success' => false, 'message' => '标题不能为空']);
        exit;
    }

    try {
        $article = TypechoPlugin\AIHelper\Plugin::generateArticle($title);

        echo json_encode([
            'success' => true,
            'content' => $article
        ]);
        exit;

    } catch (Exception $e) {
        // 返回详细的错误信息
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage(),
            'debug' => [
                'file' => basename($e->getFile()),
                'line' => $e->getLine()
            ]
        ]);
        exit;
    }
}

// 处理AJAX单篇生成摘要请求
if (isset($_POST['ajax_generate']) && $_POST['ajax_generate'] == '1') {
    header('Content-Type: application/json');

    $cid = isset($_POST['cid']) ? intval($_POST['cid']) : 0;

    if (!$cid) {
        echo json_encode(['success' => false, 'message' => '无效的文章ID']);
        exit;
    }

    try {
        // 获取文章内容
        $post = $db->fetchRow($db->select('text')->from('table.contents')
            ->where('cid = ?', $cid));

        if (!$post) {
            echo json_encode(['success' => false, 'message' => '未找到文章']);
            exit;
        }

        // 生成摘要
        $summary = TypechoPlugin\AIHelper\Plugin::generateSummary($post['text']);

        if (!$summary) {
            echo json_encode(['success' => false, 'message' => 'AI生成摘要失败，请检查API配置']);
            exit;
        }

        // 删除旧的summary字段
        $db->query($db->delete('table.fields')
            ->where('cid = ?', $cid)
            ->where('name = ?', 'summary'));

        // 插入新的summary字段
        $db->query($db->insert('table.fields')->rows([
            'cid' => $cid,
            'name' => 'summary',
            'type' => 'str',
            'str_value' => $summary,
            'int_value' => 0,
            'float_value' => 0
        ]));

        echo json_encode([
            'success' => true,
            'message' => '摘要生成成功',
            'summary' => $summary
        ]);
        exit;

    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => '错误: ' . $e->getMessage()]);
        exit;
    }
}

// 非AJAX请求，包含header等文件
if (!defined('__TYPECHO_ADMIN__')) {
    include 'common.php';
}
include 'header.php';
include 'menu.php';

// 处理批量生成摘要
if (isset($_POST['action']) && $_POST['action'] == 'batch_generate') {
    $cids = isset($_POST['cid']) ? $_POST['cid'] : [];
    $successCount = 0;
    $failCount = 0;
    $errors = [];

    foreach ($cids as $cid) {
        $cid = intval($cid);

        try {
            // 获取文章内容
            $post = $db->fetchRow($db->select('text')->from('table.contents')
                ->where('cid = ?', $cid));

            if (!$post) {
                $failCount++;
                $errors[] = "文章ID {$cid}: 未找到文章";
                continue;
            }

            // 生成摘要
            $summary = TypechoPlugin\AIHelper\Plugin::generateSummary($post['text']);

            if (!$summary) {
                $failCount++;
                $errors[] = "文章ID {$cid}: 生成摘要失败";
                continue;
            }

            // 删除旧的summary字段
            $db->query($db->delete('table.fields')
                ->where('cid = ?', $cid)
                ->where('name = ?', 'summary'));

            // 插入新的summary字段
            $db->query($db->insert('table.fields')->rows([
                'cid' => $cid,
                'name' => 'summary',
                'type' => 'str',
                'str_value' => $summary,
                'int_value' => 0,
                'float_value' => 0
            ]));

            $successCount++;
        } catch (Exception $e) {
            $failCount++;
            $errors[] = "文章ID {$cid}: " . $e->getMessage();
        }
    }

    $message = "批量生成完成！成功: {$successCount} 篇，失败: {$failCount} 篇";
}

// 获取所有文章
$posts = $db->fetchAll($db->select('cid', 'title', 'created')
    ->from('table.contents')
    ->where('type = ?', 'post')
    ->order('created', Db::SORT_DESC)
    ->limit(100));

?>

<div class="main">
    <div class="body container">
        <div class="colgroup">
            <div class="typecho-page-title col-mb-12">
                <h2>批量生成摘要</h2>
            </div>
        </div>

        <?php if (isset($message)): ?>
        <div class="col-mb-12">
            <div class="message success">
                <?php echo $message; ?>
            </div>
            <?php if (!empty($errors)): ?>
            <div class="message error">
                <h3>错误详情：</h3>
                <ul>
                    <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <div class="col-mb-12">
            <form method="post">
                <input type="hidden" name="action" value="batch_generate" />

                <div class="typecho-table-wrap">
                    <table class="typecho-list-table">
                        <colgroup>
                            <col width="20"/>
                            <col width="60%"/>
                            <col width="30%"/>
                            <col width="10%"/>
                        </colgroup>
                        <thead>
                            <tr>
                                <th><input type="checkbox" class="typecho-table-select-all" /></th>
                                <th>标题</th>
                                <th>发布时间</th>
                                <th>操作</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($posts as $post): ?>
                            <?php
                                // 检查是否已有摘要
                                $hasSummary = $db->fetchRow($db->select('str_value')
                                    ->from('table.fields')
                                    ->where('cid = ?', $post['cid'])
                                    ->where('name = ?', 'summary'));
                            ?>
                            <tr>
                                <td><input type="checkbox" name="cid[]" value="<?php echo $post['cid']; ?>" class="typecho-table-select" /></td>
                                <td>
                                    <?php echo htmlspecialchars($post['title']); ?>
                                    <?php if ($hasSummary): ?>
                                    <span style="color: #999;">(已有摘要)</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('Y-m-d H:i:s', $post['created']); ?></td>
                                <td>
                                    <a href="<?php echo \Typecho\Common::url('write-post.php?cid=' . $post['cid'], Options::alloc()->adminUrl); ?>">编辑</a> |
                                    <a href="javascript:void(0);" class="generate-single" data-cid="<?php echo $post['cid']; ?>" data-title="<?php echo htmlspecialchars($post['title']); ?>">生成摘要</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="typecho-table-select">
                    <button type="submit" class="btn primary">批量生成摘要</button>
                    <span class="description">将会覆盖已有的摘要字段</span>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
(function() {
    // 全选/取消全选
    var selectAll = document.querySelector('.typecho-table-select-all');
    var checkboxes = document.querySelectorAll('.typecho-table-select');

    if (selectAll) {
        selectAll.addEventListener('change', function() {
            checkboxes.forEach(function(checkbox) {
                checkbox.checked = selectAll.checked;
            });
        });
    }

    // 单篇文章生成摘要
    var generateButtons = document.querySelectorAll('.generate-single');

    generateButtons.forEach(function(button) {
        button.addEventListener('click', function(e) {
            e.preventDefault();

            var cid = this.getAttribute('data-cid');
            var title = this.getAttribute('data-title');
            var originalText = this.textContent;
            var link = this;

            if (!confirm('确定要为文章《' + title + '》生成摘要吗？\n如果已有摘要将会被覆盖。')) {
                return;
            }

            // 显示加载状态
            link.textContent = '生成中...';
            link.style.color = '#999';
            link.style.pointerEvents = 'none';

            // 发送AJAX请求
            var xhr = new XMLHttpRequest();
            xhr.open('POST', window.location.href, true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');

            xhr.onload = function() {
                if (xhr.status === 200) {
                    try {
                        var response = JSON.parse(xhr.responseText);

                        if (response.success) {
                            link.textContent = '生成成功';
                            link.style.color = '#467B96';

                            // 更新"已有摘要"标记
                            var titleCell = link.closest('tr').querySelector('td:nth-child(2)');
                            var hasSummarySpan = titleCell.querySelector('span');
                            if (!hasSummarySpan) {
                                var span = document.createElement('span');
                                span.style.color = '#999';
                                span.textContent = '(已有摘要)';
                                titleCell.appendChild(document.createTextNode(' '));
                                titleCell.appendChild(span);
                            }

                            // 3秒后恢复按钮
                            setTimeout(function() {
                                link.textContent = originalText;
                                link.style.color = '';
                                link.style.pointerEvents = '';
                            }, 3000);

                        } else {
                            alert('生成失败: ' + response.message);
                            link.textContent = originalText;
                            link.style.color = '';
                            link.style.pointerEvents = '';
                        }
                    } catch (e) {
                        alert('响应解析错误');
                        link.textContent = originalText;
                        link.style.color = '';
                        link.style.pointerEvents = '';
                    }
                } else {
                    alert('请求失败: HTTP ' + xhr.status);
                    link.textContent = originalText;
                    link.style.color = '';
                    link.style.pointerEvents = '';
                }
            };

            xhr.onerror = function() {
                alert('网络错误');
                link.textContent = originalText;
                link.style.color = '';
                link.style.pointerEvents = '';
            };

            xhr.send('ajax_generate=1&cid=' + encodeURIComponent(cid));
        });
    });
})();
</script>

<?php
include 'copyright.php';
include 'common-js.php';
include 'footer.php';
?>
