<?php

namespace TypechoPlugin\AIHelper;

use Typecho\Plugin\PluginInterface;
use Typecho\Widget\Helper\Form;
use Typecho\Widget\Helper\Form\Element\Text;
use Typecho\Widget\Helper\Form\Element\Textarea;
use Widget\Options;
use Typecho\Db;
use Typecho\Common;
use Widget\Base\Contents;

if (!defined('__TYPECHO_ROOT_DIR__')) {
    exit;
}

/**
 * AI助手插件 - 自动生成Slug、摘要、Tags、文章
 *
 * @package AIHelper
 * @author AI
 * @version 1.0.1
 * @link https://github.com
 */
class Plugin implements PluginInterface
{
    /**
     * 激活插件方法
     */
    public static function activate()
    {
        // 挂载到文章写入前的钩子，用于自动生成slug
        \Typecho\Plugin::factory('Widget_Contents_Post_Edit')->write = __CLASS__ . '::onPostWrite';
        \Typecho\Plugin::factory('Widget_Contents_Page_Edit')->write = __CLASS__ . '::onPostWrite';

        // 挂载到文章发布完成后的钩子，用于生成摘要和标签
        \Typecho\Plugin::factory('Widget_Contents_Post_Edit')->finishPublish = __CLASS__ . '::onFinishPublish';
        \Typecho\Plugin::factory('Widget_Contents_Page_Edit')->finishPublish = __CLASS__ . '::onFinishPublish';

        // 挂载到后台管理页面的钩子，用于添加脚本和按钮
        \Typecho\Plugin::factory('admin/write-post.php')->bottom = __CLASS__ . '::addAdminScript';

        // 在"管理"菜单下注册"批量摘要"面板
        \Utils\Helper::addPanel(2, 'AIHelper/Panel.php', '批量摘要', '批量生成文章摘要', 'administrator');

        return _t('插件已激活，请配置API设置');
    }

    /**
     * 禁用插件方法
     */
    public static function deactivate()
    {
        // 清理面板注册
        $options = \Widget\Options::alloc();
        $panelTable = unserialize($options->panelTable);

        if (isset($panelTable['file'])) {
            // 移除AIHelper的面板文件
            $panelTable['file'] = array_filter($panelTable['file'], function($file) {
                return strpos(urldecode($file), 'AIHelper/') !== 0;
            });
        }

        if (isset($panelTable['child'][2])) {
            // 移除管理菜单下的AIHelper面板
            $panelTable['child'][2] = array_filter($panelTable['child'][2], function($item) {
                return !isset($item[2]) || strpos($item[2], 'AIHelper') === false;
            });
            // 重新索引数组
            $panelTable['child'][2] = array_values($panelTable['child'][2]);
        }

        // 更新配置
        $options->update(['panelTable' => serialize($panelTable)], \Typecho\Db::get()->sql()->where('name = ?', 'panelTable'));

        return _t('插件已禁用');
    }

    /**
     * 获取插件配置面板
     *
     * @param Form $form 配置面板
     */
    public static function config(Form $form)
    {
        $apiUrl = new Text('apiUrl', null, 'https://api.deepseek.com',
            _t('API地址'), _t('填写兼容OpenAI格式的API地址，如：https://api.deepseek.com'));
        $form->addInput($apiUrl->addRule('required', _t('API地址不能为空')));

        $apiKey = new Text('apiKey', null, '',
            _t('API密钥'), _t('填写您的API密钥'));
        $form->addInput($apiKey->addRule('required', _t('API密钥不能为空')));

        $model = new Text('model', null, 'deepseek-chat',
            _t('模型名称'), _t('填写模型名称，如：deepseek-chat, gpt-3.5-turbo等'));
        $form->addInput($model->addRule('required', _t('模型名称不能为空')));

        $slugPrompt = new Textarea('slugPrompt', null,
            '请将以下中文标题翻译成符合英文语法的简短slug，使用连字符(-)连接单词，只返回slug本身，不要有其他内容。标题：{title}',
            _t('Slug生成提示词'), _t('使用{title}作为标题占位符'));
        $form->addInput($slugPrompt);

        $summaryPrompt = new Textarea('summaryPrompt', null,
            '请为以下文章生成一个简洁的摘要(150字以内)：\n\n{content}',
            _t('摘要生成提示词'), _t('使用{content}作为内容占位符'));
        $form->addInput($summaryPrompt);

        $tagsPrompt = new Textarea('tagsPrompt', null,
            '请为以下文章提取1-5个最相关的标签，直接返回标签，用逗号分隔，不要有其他内容：\n\n{content}',
            _t('标签提取提示词'), _t('使用{content}作为内容占位符'));
        $form->addInput($tagsPrompt);

        $articlePrompt = new Textarea('articlePrompt', null,
            '请根据以下标题撰写一篇约1000字的文章。要求内容充实、结构清晰、语言流畅。标题：{title}',
            _t('文章生成提示词'), _t('使用{title}作为标题占位符，用于自动生成文章内容'));
        $form->addInput($articlePrompt);
    }

    /**
     * 个人用户的配置面板
     *
     * @param Form $form
     */
    public static function personalConfig(Form $form)
    {
    }

    /**
     * 调用AI API
     */
    private static function callAI($prompt, $temperature = 0.7, $maxTokens = 1000)
    {
        $options = Options::alloc()->plugin('AIHelper');

        $apiUrl = rtrim($options->apiUrl, '/') . '/v1/chat/completions';
        $apiKey = $options->apiKey;
        $model = $options->model;

        $data = [
            'model' => $model,
            'messages' => [
                ['role' => 'user', 'content' => $prompt]
            ],
            'temperature' => $temperature,
            'max_tokens' => $maxTokens
        ];

        $ch = curl_init($apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiKey
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60); // 增加超时时间到60秒

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            throw new \Exception('网络请求失败: ' . $curlError);
        }

        if ($httpCode !== 200) {
            // 记录错误响应
            @file_put_contents(
                __DIR__ . '/debug.log',
                date('Y-m-d H:i:s') . " - API Error: HTTP $httpCode\nResponse: $response\n",
                FILE_APPEND
            );
            throw new \Exception('API调用失败: HTTP ' . $httpCode);
        }

        $result = json_decode($response, true);

        if (isset($result['choices'][0]['message']['content'])) {
            return trim($result['choices'][0]['message']['content']);
        }

        throw new \Exception('API返回格式错误');
    }

    /**
     * 生成Slug
     */
    public static function generateSlug($title)
    {
        try {
            $options = Options::alloc()->plugin('AIHelper');
            $prompt = str_replace('{title}', $title, $options->slugPrompt);
            $slug = self::callAI($prompt, 0.3);

            // 清理slug，确保只包含字母、数字和连字符
            $slug = preg_replace('/[^a-z0-9-]+/', '', strtolower($slug));
            $slug = preg_replace('/-+/', '-', $slug);
            $slug = trim($slug, '-');

            return $slug ?: 'untitled';
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * 生成摘要
     */
    public static function generateSummary($content)
    {
        try {
            $options = Options::alloc()->plugin('AIHelper');

            // 移除HTML标签和多余空白
            $cleanContent = strip_tags($content);
            $cleanContent = preg_replace('/\s+/', ' ', $cleanContent);
            $cleanContent = mb_substr($cleanContent, 0, 3000); // 限制长度

            $prompt = str_replace('{content}', $cleanContent, $options->summaryPrompt);
            return self::callAI($prompt, 0.7);
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * 提取标签
     */
    public static function extractTags($content)
    {
        try {
            $options = Options::alloc()->plugin('AIHelper');

            // 移除HTML标签和多余空白
            $cleanContent = strip_tags($content);
            $cleanContent = preg_replace('/\s+/', ' ', $cleanContent);
            $cleanContent = mb_substr($cleanContent, 0, 2000); // 限制长度

            $prompt = str_replace('{content}', $cleanContent, $options->tagsPrompt);
            $tagsStr = self::callAI($prompt, 0.5);

            // 解析标签
            $tags = array_map('trim', explode(',', $tagsStr));
            $tags = array_filter($tags);
            $tags = array_slice($tags, 0, 5); // 最多5个

            return $tags;
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * 根据标题生成文章内容
     */
    public static function generateArticle($title)
    {
        try {
            $options = Options::alloc()->plugin('AIHelper');

            if (empty($title)) {
                throw new \Exception('标题不能为空');
            }

            $prompt = str_replace('{title}', $title, $options->articlePrompt);

            // 生成文章需要更多tokens，设置为3000（约1500字）
            $article = self::callAI($prompt, 0.8, 3000);

            if (empty($article)) {
                throw new \Exception('AI返回内容为空');
            }

            return $article;
        } catch (\Exception $e) {
            // 记录错误日志
            @file_put_contents(
                __DIR__ . '/debug.log',
                date('Y-m-d H:i:s') . " - generateArticle Error: " . $e->getMessage() . "\n",
                FILE_APPEND
            );
            throw $e; // 重新抛出异常，让调用者知道错误
        }
    }

    /**
     * 文章写入前触发 - 自动生成slug
     */
    public static function onPostWrite($contents, $edit)
    {
        // 如果slug为空，自动生成
        if (empty($contents['slug']) && !empty($contents['title'])) {
            try {
                $slug = self::generateSlug($contents['title']);
                if ($slug) {
                    $contents['slug'] = $slug;
                }
            } catch (\Exception $e) {
                // 如果生成失败，使用默认的slug生成方式
                // 不中断流程
            }
        }

        return $contents;
    }

    /**
     * 文章发布完成后触发 - 生成摘要和标签
     */
    public static function onFinishPublish($contents, $edit)
    {
        // 记录钩子被触发
        @file_put_contents(
            __DIR__ . '/debug.log',
            date('Y-m-d H:i:s') . " - Hook triggered\n",
            FILE_APPEND
        );

        try {
            // 调试：记录$contents和$edit的内容
            @file_put_contents(
                __DIR__ . '/debug.log',
                date('Y-m-d H:i:s') . " - contents type: " . gettype($contents) . "\n",
                FILE_APPEND
            );

            @file_put_contents(
                __DIR__ . '/debug.log',
                date('Y-m-d H:i:s') . " - edit type: " . gettype($edit) . "\n",
                FILE_APPEND
            );

            // 尝试从$edit对象中获取cid
            $cid = null;
            $text = null;

            // $edit可能是Widget对象
            if (is_object($edit)) {
                @file_put_contents(
                    __DIR__ . '/debug.log',
                    date('Y-m-d H:i:s') . " - edit class: " . get_class($edit) . "\n",
                    FILE_APPEND
                );

                // 尝试获取cid
                if (isset($edit->cid)) {
                    $cid = $edit->cid;
                } elseif (method_exists($edit, 'cid')) {
                    $cid = $edit->cid;
                } elseif (isset($edit->request) && isset($edit->request->cid)) {
                    $cid = $edit->request->cid;
                }

                // 尝试获取text
                if (isset($edit->text)) {
                    $text = $edit->text;
                } elseif (method_exists($edit, 'text')) {
                    $text = $edit->text;
                }
            }

            // 如果$contents是数组，也尝试从中获取
            if (is_array($contents)) {
                if (!empty($contents['cid'])) {
                    $cid = $contents['cid'];
                }
                if (!empty($contents['text'])) {
                    $text = $contents['text'];
                }
            }

            if (empty($cid)) {
                @file_put_contents(
                    __DIR__ . '/debug.log',
                    date('Y-m-d H:i:s') . " - No cid found\n",
                    FILE_APPEND
                );
                return $contents;
            }

            @file_put_contents(
                __DIR__ . '/debug.log',
                date('Y-m-d H:i:s') . " - Processing cid: {$cid}\n",
                FILE_APPEND
            );

            $db = Db::get();

            // 检查是否已有summary字段，如果有则不生成
            $existingFields = $db->fetchRow($db->select('str_value')
                ->from('table.fields')
                ->where('cid = ?', $cid)
                ->where('name = ?', 'summary'));

            if (!empty($existingFields)) {
                @file_put_contents(
                    __DIR__ . '/debug.log',
                    date('Y-m-d H:i:s') . " - Summary already exists\n",
                    FILE_APPEND
                );
                return $contents;
            }

            // 如果没有从参数获取到text，从数据库查询
            if (empty($text)) {
                $post = $db->fetchRow($db->select('text')->from('table.contents')
                    ->where('cid = ?', $cid));
                if ($post) {
                    $text = $post['text'];
                }
            }

            // 生成摘要和标签
            if (!empty($text)) {
                // 生成摘要
                try {
                    @file_put_contents(
                        __DIR__ . '/debug.log',
                        date('Y-m-d H:i:s') . " - Generating summary...\n",
                        FILE_APPEND
                    );

                    $summary = self::generateSummary($text);

                    // 保存摘要到自定义字段
                    if ($summary) {
                        @file_put_contents(
                            __DIR__ . '/debug.log',
                            date('Y-m-d H:i:s') . " - Summary generated: " . substr($summary, 0, 50) . "...\n",
                            FILE_APPEND
                        );

                        // 先删除可能存在的旧记录（避免重复）
                        $db->query($db->delete('table.fields')
                            ->where('cid = ?', $cid)
                            ->where('name = ?', 'summary'));

                        // 插入新记录
                        $insertData = [
                            'cid' => $cid,
                            'name' => 'summary',
                            'type' => 'str',
                            'str_value' => $summary,
                            'int_value' => 0,
                            'float_value' => 0
                        ];

                        $db->query($db->insert('table.fields')->rows($insertData));

                        // 写入日志
                        @file_put_contents(
                            __DIR__ . '/debug.log',
                            date('Y-m-d H:i:s') . " - Summary saved for cid: {$cid}\n",
                            FILE_APPEND
                        );
                    } else {
                        @file_put_contents(
                            __DIR__ . '/debug.log',
                            date('Y-m-d H:i:s') . " - Summary is empty\n",
                            FILE_APPEND
                        );
                    }
                } catch (\Exception $e) {
                    // 记录错误日志
                    @file_put_contents(
                        __DIR__ . '/debug.log',
                        date('Y-m-d H:i:s') . " - Summary error: " . $e->getMessage() . "\n",
                        FILE_APPEND
                    );
                }

                // 提取并保存标签
                try {
                    $tags = self::extractTags($text);
                    if (!empty($tags)) {
                        self::saveTags($cid, $tags);
                        @file_put_contents(
                            __DIR__ . '/debug.log',
                            date('Y-m-d H:i:s') . " - Tags saved: " . implode(', ', $tags) . "\n",
                            FILE_APPEND
                        );
                    }
                } catch (\Exception $e) {
                    // 标签提取失败，不影响发布
                    @file_put_contents(
                        __DIR__ . '/debug.log',
                        date('Y-m-d H:i:s') . " - Tags error: " . $e->getMessage() . "\n",
                        FILE_APPEND
                    );
                }
            } else {
                @file_put_contents(
                    __DIR__ . '/debug.log',
                    date('Y-m-d H:i:s') . " - No text found\n",
                    FILE_APPEND
                );
            }
        } catch (\Exception $e) {
            // 捕获所有异常，不影响文章发布
            @file_put_contents(
                __DIR__ . '/debug.log',
                date('Y-m-d H:i:s') . " - General error: " . $e->getMessage() . "\n",
                FILE_APPEND
            );
        }

        return $contents;
    }

    /**
     * 保存标签
     */
    private static function saveTags($cid, $tags)
    {
        try {
            $db = Db::get();

            foreach ($tags as $tagName) {
                $tagName = trim($tagName);
                if (empty($tagName)) {
                    continue;
                }

                try {
                    // 检查标签是否存在
                    $tag = $db->fetchRow($db->select('mid')->from('table.metas')
                        ->where('name = ?', $tagName)
                        ->where('type = ?', 'tag'));

                    if ($tag) {
                        $mid = $tag['mid'];
                    } else {
                        // 创建新标签
                        $slug = \Typecho\Common::slugName($tagName);

                        // 检查slug是否已存在，如果存在则添加后缀
                        $slugExists = $db->fetchRow($db->select('mid')->from('table.metas')
                            ->where('slug = ?', $slug)
                            ->where('type = ?', 'tag'));

                        if ($slugExists) {
                            $slug = $slug . '-' . time();
                        }

                        $insertId = $db->query($db->insert('table.metas')->rows([
                            'name' => $tagName,
                            'slug' => $slug,
                            'type' => 'tag',
                            'description' => '',
                            'count' => 0,
                            'order' => 0,
                            'parent' => 0
                        ]));

                        // 获取刚插入的标签ID
                        $newTag = $db->fetchRow($db->select('mid')->from('table.metas')
                            ->where('slug = ?', $slug)
                            ->where('type = ?', 'tag')
                            ->order('mid', Db::SORT_DESC)
                            ->limit(1));

                        $mid = $newTag ? $newTag['mid'] : 0;

                        if (!$mid) {
                            continue; // 如果无法获取ID，跳过这个标签
                        }
                    }

                    // 检查是否已经关联
                    $relationship = $db->fetchRow($db->select()->from('table.relationships')
                        ->where('cid = ?', $cid)
                        ->where('mid = ?', $mid));

                    if (!$relationship) {
                        // 关联标签到文章
                        $db->query($db->insert('table.relationships')->rows([
                            'cid' => $cid,
                            'mid' => $mid
                        ]));

                        // 更新标签计数
                        $currentCount = $db->fetchObject($db->select('count')->from('table.metas')
                            ->where('mid = ?', $mid));

                        if ($currentCount) {
                            $db->query($db->update('table.metas')
                                ->rows(['count' => $currentCount->count + 1])
                                ->where('mid = ?', $mid));
                        }
                    }
                } catch (\Exception $e) {
                    // 单个标签失败不影响其他标签
                    continue;
                }
            }
        } catch (\Exception $e) {
            // 保存标签失败不影响文章发布
        }
    }

    /**
     * 在文章编辑页面底部添加脚本和按钮
     */
    public static function addAdminScript()
    {
        $pluginUrl = \Typecho\Common::url('usr/plugins/AIHelper', Options::alloc()->siteUrl);
        $panelUrl = \Typecho\Common::url('extending.php?panel=AIHelper/Panel.php', Options::alloc()->adminUrl);

        echo '<style>
        #aihelper-generate-btn {
            margin-top: 10px;
            width: 100%;
            background: #467B96;
            color: white;
            border: none;
            padding: 8px 15px;
            cursor: pointer;
            border-radius: 2px;
            font-size: 14px;
        }
        #aihelper-generate-btn:hover {
            background: #356482;
        }
        #aihelper-generate-btn:disabled {
            background: #ccc;
            cursor: not-allowed;
        }

        /* Toast通知样式 */
        .aihelper-toast {
            position: fixed;
            top: 80px;
            right: 30px;
            min-width: 300px;
            max-width: 500px;
            background: white;
            border-radius: 4px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            padding: 16px 20px;
            z-index: 9999;
            animation: slideIn 0.3s ease-out;
            display: flex;
            align-items: flex-start;
            gap: 12px;
        }

        @keyframes slideIn {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        @keyframes slideOut {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(400px);
                opacity: 0;
            }
        }

        .aihelper-toast.hide {
            animation: slideOut 0.3s ease-out forwards;
        }

        .aihelper-toast-icon {
            flex-shrink: 0;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            font-weight: bold;
        }

        .aihelper-toast.success .aihelper-toast-icon {
            background: #52c41a;
            color: white;
        }

        .aihelper-toast.error .aihelper-toast-icon {
            background: #ff4d4f;
            color: white;
        }

        .aihelper-toast.loading .aihelper-toast-icon {
            background: #1890ff;
            color: white;
            animation: rotate 1s linear infinite;
        }

        @keyframes rotate {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        .aihelper-toast-content {
            flex: 1;
        }

        .aihelper-toast-title {
            font-weight: 500;
            margin-bottom: 4px;
            color: #262626;
        }

        .aihelper-toast-message {
            font-size: 13px;
            color: #8c8c8c;
            line-height: 1.5;
        }

        .aihelper-toast-close {
            flex-shrink: 0;
            cursor: pointer;
            color: #8c8c8c;
            font-size: 18px;
            line-height: 1;
            padding: 0 4px;
        }

        .aihelper-toast-close:hover {
            color: #262626;
        }

        /* 确认对话框样式 */
        .aihelper-modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.45);
            z-index: 9998;
            display: flex;
            align-items: center;
            justify-content: center;
            animation: fadeIn 0.2s ease-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .aihelper-modal {
            background: white;
            border-radius: 4px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            min-width: 400px;
            max-width: 520px;
            animation: modalSlideIn 0.3s ease-out;
        }

        @keyframes modalSlideIn {
            from {
                transform: scale(0.9);
                opacity: 0;
            }
            to {
                transform: scale(1);
                opacity: 1;
            }
        }

        .aihelper-modal-header {
            padding: 16px 24px;
            border-bottom: 1px solid #f0f0f0;
            font-weight: 500;
            font-size: 16px;
        }

        .aihelper-modal-body {
            padding: 24px;
            color: #595959;
            line-height: 1.6;
        }

        .aihelper-modal-footer {
            padding: 10px 16px;
            border-top: 1px solid #f0f0f0;
            text-align: right;
        }

        .aihelper-modal-btn {
            padding: 6px 16px;
            border-radius: 2px;
            border: 1px solid #d9d9d9;
            background: white;
            cursor: pointer;
            margin-left: 8px;
            font-size: 14px;
        }

        .aihelper-modal-btn:hover {
            border-color: #40a9ff;
            color: #40a9ff;
        }

        .aihelper-modal-btn.primary {
            background: #467B96;
            border-color: #467B96;
            color: white;
        }

        .aihelper-modal-btn.primary:hover {
            background: #356482;
            border-color: #356482;
        }
        </style>';

        echo '<script>
        (function() {
            // 防止重复执行
            if (window.aihelperLoaded) {
                console.log("[AIHelper] 已加载，跳过重复执行");
                return;
            }
            window.aihelperLoaded = true;

            // Toast通知函数
            window.aihelperToast = function(options) {
                var type = options.type || "success"; // success, error, loading
                var title = options.title || "";
                var message = options.message || "";
                var duration = options.duration || 3000;
                var onClose = options.onClose || null;

                // 创建toast元素
                var toast = document.createElement("div");
                toast.className = "aihelper-toast " + type;

                var icon = type === "success" ? "✓" : (type === "error" ? "✕" : "⟳");

                toast.innerHTML = `
                    <div class="aihelper-toast-icon">${icon}</div>
                    <div class="aihelper-toast-content">
                        ${title ? `<div class="aihelper-toast-title">${title}</div>` : ""}
                        ${message ? `<div class="aihelper-toast-message">${message}</div>` : ""}
                    </div>
                    ${type !== "loading" ? `<span class="aihelper-toast-close">×</span>` : ""}
                `;

                document.body.appendChild(toast);

                var closeBtn = toast.querySelector(".aihelper-toast-close");
                if (closeBtn) {
                    closeBtn.onclick = function() {
                        closeToast();
                    };
                }

                function closeToast() {
                    toast.classList.add("hide");
                    setTimeout(function() {
                        if (toast.parentNode) {
                            toast.parentNode.removeChild(toast);
                        }
                        if (onClose) onClose();
                    }, 300);
                }

                // 自动关闭（loading类型不自动关闭）
                if (type !== "loading" && duration > 0) {
                    setTimeout(closeToast, duration);
                }

                return {
                    close: closeToast,
                    element: toast
                };
            };

            // 确认对话框
            window.aihelperConfirm = function(options) {
                return new Promise(function(resolve) {
                    var title = options.title || "确认";
                    var content = options.content || "";
                    var okText = options.okText || "确定";
                    var cancelText = options.cancelText || "取消";

                    var overlay = document.createElement("div");
                    overlay.className = "aihelper-modal-overlay";

                    overlay.innerHTML = `
                        <div class="aihelper-modal">
                            <div class="aihelper-modal-header">${title}</div>
                            <div class="aihelper-modal-body">${content}</div>
                            <div class="aihelper-modal-footer">
                                <button class="aihelper-modal-btn cancel">${cancelText}</button>
                                <button class="aihelper-modal-btn primary ok">${okText}</button>
                            </div>
                        </div>
                    `;

                    document.body.appendChild(overlay);

                    function close(result) {
                        overlay.style.animation = "fadeOut 0.2s ease-out";
                        setTimeout(function() {
                            if (overlay.parentNode) {
                                overlay.parentNode.removeChild(overlay);
                            }
                            resolve(result);
                        }, 200);
                    }

                    overlay.querySelector(".ok").onclick = function() { close(true); };
                    overlay.querySelector(".cancel").onclick = function() { close(false); };
                    overlay.onclick = function(e) {
                        if (e.target === overlay) close(false);
                    };
                });
            };

            jQuery(document).ready(function($) {
                console.log("[AIHelper] 脚本已加载");

                // 检查按钮是否已存在
                if ($("#aihelper-generate-btn").length > 0) {
                    console.log("[AIHelper] 按钮已存在，跳过");
                    return;
                }

                // 尝试多种选择器查找合适的插入位置
                var insertTarget = null;

                // 尝试1: 查找预览按钮容器
                var previewContainer = $(".submit .right, .submit .operate, .typecho-post-option .submit");
                console.log("[AIHelper] 找到预览容器:", previewContainer.length);

                if (previewContainer.length > 0) {
                    insertTarget = previewContainer.first();
                } else {
                    // 尝试2: 查找提交按钮的父容器
                    var submitBtn = $("button[type=submit], input[type=submit]").first();
                    if (submitBtn.length > 0) {
                        insertTarget = submitBtn.parent();
                        console.log("[AIHelper] 使用提交按钮父容器");
                    }
                }

                if (insertTarget && insertTarget.length > 0) {
                    var generateBtn = $(\'<button type="button" id="aihelper-generate-btn">自动生成文章</button>\');
                    insertTarget.append(generateBtn);
                    console.log("[AIHelper] 按钮已添加");

                    generateBtn.on("click", function() {
                        var title = $("#title, input[name=title]").val();
                        console.log("[AIHelper] 标题:", title);

                        if (!title || title.trim() === "") {
                            aihelperToast({
                                type: "error",
                                title: "请先输入文章标题",
                                duration: 2000
                            });
                            return;
                        }

                        aihelperConfirm({
                            title: "生成文章确认",
                            content: `确定要根据标题《${title}》自动生成文章内容吗？<br><br><strong>注意：</strong>生成的内容将覆盖当前编辑器中的内容。`,
                            okText: "开始生成",
                            cancelText: "取消"
                        }).then(function(confirmed) {
                            if (!confirmed) return;

                            // 禁用按钮
                            generateBtn.prop("disabled", true).text("生成中...");

                            // 显示加载提示
                            var loadingToast = aihelperToast({
                                type: "loading",
                                title: "正在生成文章...",
                                message: "这可能需要几秒到十几秒，请耐心等待",
                                duration: 0
                            });

                            // 发送AJAX请求
                            $.ajax({
                                url: "' . $panelUrl . '",
                                type: "POST",
                                data: {
                                    ajax_action: "generate_article",
                                    title: title
                                },
                                dataType: "json",
                                success: function(response) {
                                    console.log("[AIHelper] 响应:", response);
                                    loadingToast.close();

                                    if (response.success) {
                                        // 填充到编辑器
                                        var editor = $("#text, textarea[name=text]");
                                        console.log("[AIHelper] 找到编辑器:", editor.length);

                                        if (editor.length > 0) {
                                            // 检查是否有CodeMirror
                                            if (editor[0].CodeMirror) {
                                                editor[0].CodeMirror.setValue(response.content);
                                                console.log("[AIHelper] 已填充到CodeMirror");
                                            } else {
                                                editor.val(response.content);
                                                console.log("[AIHelper] 已填充到textarea");
                                            }
                                        }

                                        aihelperToast({
                                            type: "success",
                                            title: "文章生成成功！",
                                            message: "内容已填充到编辑器，请查看并根据需要修改",
                                            duration: 4000
                                        });
                                    } else {
                                        aihelperToast({
                                            type: "error",
                                            title: "生成失败",
                                            message: response.message || "未知错误",
                                            duration: 5000
                                        });
                                    }
                                },
                                error: function(xhr, status, error) {
                                    console.error("[AIHelper] AJAX错误:", error);
                                    loadingToast.close();

                                    aihelperToast({
                                        type: "error",
                                        title: "请求失败",
                                        message: "网络错误或服务器无响应，请稍后重试",
                                        duration: 5000
                                    });
                                },
                                complete: function() {
                                    generateBtn.prop("disabled", false).text("自动生成文章");
                                }
                            });
                        });
                    });
                } else {
                    console.error("[AIHelper] 未找到合适的插入位置");
                }
            });
        })();
        </script>';
    }

    /**
     * 在文章管理页面添加批量操作
     */
    public static function addBatchAction()
    {
        $pluginUrl = \Typecho\Common::url('usr/plugins/AIHelper', Options::alloc()->siteUrl);
        echo '<script src="' . $pluginUrl . '/static/batch.js?v=' . time() . '"></script>';
    }

    /**
     * 添加批量生成摘要链接
     */
    public static function addSummaryLink()
    {
        $panelUrl = \Typecho\Common::url('extending.php?panel=AIHelper/Panel.php', Options::alloc()->adminUrl);
        echo '<style>.aihelper-link { margin-left: 10px; padding: 5px 10px; background: #467B96; color: white; border-radius: 3px; text-decoration: none; } .aihelper-link:hover { background: #5A95B8; color: white; }</style>';
        echo '<script>
        jQuery(document).ready(function($) {
            $(".typecho-page-title h2").append(\'<a href="' . $panelUrl . '" class="aihelper-link">批量生成摘要</a>\');
        });
        </script>';
    }

    /**
     * 在左侧菜单添加摘要菜单项
     */
    public static function addMenuItem()
    {
        $panelUrl = \Typecho\Common::url('extending.php?panel=AIHelper/Panel.php', Options::alloc()->adminUrl);
        echo '<script>
        (function() {
            if (typeof jQuery !== "undefined") {
                jQuery(document).ready(function($) {
                    console.log("[AIHelper] 开始添加菜单项...");

                    // 尝试多种选择器
                    var $menu = $("#typecho-nav-list, .typecho-head-nav, #typecho-option-tabs, ul.root");
                    console.log("[AIHelper] 找到的菜单容器:", $menu.length);

                    if ($menu.length === 0) {
                        // 如果没找到，遍历所有可能的导航列表
                        $menu = $("nav ul, aside ul, .menu ul").first();
                        console.log("[AIHelper] 使用备用选择器，找到:", $menu.length);
                    }

                    // 打印菜单HTML结构以便调试
                    if ($menu.length > 0) {
                        console.log("[AIHelper] 菜单HTML:", $menu.html().substring(0, 200));
                    }

                    // 查找包含"管理"或"manage"的链接
                    var $manageLink = $menu.find("a:contains(\'管理\'), a[href*=\'manage\']").first();
                    console.log("[AIHelper] 找到管理链接:", $manageLink.length);

                    if ($manageLink.length > 0) {
                        var $manageItem = $manageLink.parent();
                        var $subMenu = $manageItem.find("ul").first();

                        if ($subMenu.length === 0) {
                            $subMenu = $manageItem.next("ul");
                        }

                        console.log("[AIHelper] 找到子菜单:", $subMenu.length);

                        if ($subMenu.length > 0) {
                            // 检查是否已存在
                            if ($subMenu.find("a[href*=\'AIHelper\']").length === 0) {
                                var $summaryItem = $(\'<li><a href="' . $panelUrl . '">摘要</a></li>\');
                                $subMenu.append($summaryItem);
                                console.log("[AIHelper] 菜单项已添加");
                            } else {
                                console.log("[AIHelper] 菜单项已存在，跳过");
                            }
                        } else {
                            console.log("[AIHelper] 未找到子菜单");
                        }
                    } else {
                        // 如果找不到管理菜单，直接在扩展菜单中添加
                        console.log("[AIHelper] 未找到管理菜单，尝试添加到扩展菜单");
                        var $extendMenu = $menu.find("a:contains(\'扩展\'), a[href*=\'extend\']").parent().next("ul");
                        if ($extendMenu.length > 0) {
                            var $summaryItem = $(\'<li><a href="' . $panelUrl . '">批量摘要</a></li>\');
                            $extendMenu.append($summaryItem);
                            console.log("[AIHelper] 已添加到扩展菜单");
                        }
                    }
                });
            } else {
                console.log("[AIHelper] jQuery未加载");
            }
        })();
        </script>';
    }
}
